import Image from 'next/image'
import Link from 'next/link'
import { Metadata } from 'next'

export const metadata: Metadata = {
  title: 'Auto Insurance Blog - Expert Tips & Guides | SecureAuto Hub',
  description: 'Read expert auto insurance articles, money-saving tips, coverage guides, and industry insights. Stay informed with SecureAuto Hub\'s comprehensive blog.',
  keywords: [
    'auto insurance blog',
    'car insurance tips',
    'insurance guides',
    'auto insurance articles',
    'insurance money saving tips',
    'coverage guides'
  ],
  openGraph: {
    title: 'Auto Insurance Blog - Expert Tips & Guides',
    description: 'Expert insights, tips, and guides to help you navigate the world of auto insurance.',
    type: 'website',
  },
}

export default function BlogPage() {
  const featuredPost = {
    title: 'The Complete Guide to Auto Insurance in 2025',
    excerpt: 'Everything you need to know about auto insurance coverage, costs, and how to find the best policy for your needs.',
    image: 'https://images.unsplash.com/photo-1549317336-206569e8475c?w=800&h=400&fit=crop',
    author: 'Sarah Johnson',
    date: 'January 15, 2025',
    readTime: '8 min read',
    slug: 'complete-guide-auto-insurance-2025',
    category: 'Insurance Basics'
  }

  const blogPosts = [
    {
      title: 'How Your Credit Score Affects Your Insurance Rates',
      excerpt: 'Learn how your credit score impacts your auto insurance premiums and steps you can take to improve both.',
      image: 'https://images.unsplash.com/photo-1554224155-6726b3ff858f?w=400&h=250&fit=crop',
      author: 'Michael Chen',
      date: 'January 12, 2025',
      readTime: '5 min read',
      slug: 'credit-score-insurance-rates',
      category: 'Money Saving Tips'
    },
    {
      title: 'What to Do After a Car Accident: A Step-by-Step Guide',
      excerpt: 'A comprehensive checklist to help you handle the aftermath of an auto accident safely and effectively.',
      image: 'https://images.unsplash.com/photo-1558618666-fcd25c85cd64?w=400&h=250&fit=crop',
      author: 'Emily Rodriguez',
      date: 'January 10, 2025',
      readTime: '6 min read',
      slug: 'what-to-do-after-car-accident',
      category: 'Claims & Safety'
    },
    {
      title: 'Understanding Deductibles: How to Choose the Right Amount',
      excerpt: 'Learn how insurance deductibles work and how to choose the right amount to balance savings with financial protection.',
      image: 'https://images.unsplash.com/photo-1554224154-26032ffc0d07?w=400&h=250&fit=crop',
      author: 'Sarah Johnson',
      date: 'January 8, 2025',
      readTime: '4 min read',
      slug: 'understanding-insurance-deductibles',
      category: 'Insurance Basics'
    },
    {
      title: '10 Ways to Lower Your Auto Insurance Premiums',
      excerpt: 'Discover proven strategies to reduce your auto insurance costs without sacrificing necessary coverage.',
      image: 'https://images.unsplash.com/photo-1579621970563-ebec7560ff3e?w=400&h=250&fit=crop',
      author: 'Michael Chen',
      date: 'January 5, 2025',
      readTime: '7 min read',
      slug: 'lower-auto-insurance-premiums',
      category: 'Money Saving Tips'
    },
    {
      title: 'Teen Driver Insurance: What Parents Need to Know',
      excerpt: 'Essential information for parents adding teen drivers to their auto insurance policy.',
      image: 'https://images.unsplash.com/photo-1449824913935-59a10b8d2000?w=400&h=250&fit=crop',
      author: 'Emily Rodriguez',
      date: 'January 3, 2025',
      readTime: '6 min read',
      slug: 'teen-driver-insurance-guide',
      category: 'Family Coverage'
    },
    {
      title: 'Gap Insurance: Do You Need It?',
      excerpt: 'Understanding gap insurance and when it makes sense to add this coverage to your auto policy.',
      image: 'https://images.unsplash.com/photo-1485291571150-772bcfc10da5?w=400&h=250&fit=crop',
      author: 'Sarah Johnson',
      date: 'December 30, 2024',
      readTime: '5 min read',
      slug: 'gap-insurance-explained',
      category: 'Coverage Options'
    }
  ]

  const categories = [
    'All Posts',
    'Insurance Basics',
    'Money Saving Tips',
    'Claims & Safety',
    'Coverage Options',
    'Family Coverage'
  ]

  return (
    <div className="min-h-screen bg-gray-50">
      {/* Hero Section */}
      <div className="bg-white">
        <div className="max-w-6xl mx-auto px-4 py-16">
          <div className="text-center mb-12">
            <h1 className="text-4xl font-bold text-gray-900 mb-4">
              SecureAuto Hub Blog
            </h1>
            <p className="text-xl text-gray-600 max-w-2xl mx-auto">
              Expert insights, tips, and guides to help you navigate the world of auto insurance
            </p>
          </div>

          {/* Featured Post */}
          <div className="mb-16">
            <div className="bg-white rounded-lg shadow-lg overflow-hidden">
              <div className="md:flex">
                <div className="md:w-1/2">
                  <Image
                    src={featuredPost.image}
                    alt={featuredPost.title}
                    width={800}
                    height={400}
                    className="w-full h-64 md:h-full object-cover"
                  />
                </div>
                <div className="md:w-1/2 p-8">
                  <div className="mb-4">
                    <span className="bg-blue-100 text-blue-800 text-sm font-medium px-3 py-1 rounded-full">
                      Featured Article
                    </span>
                  </div>
                  <h2 className="text-2xl font-bold text-gray-900 mb-4">
                    <Link href={`/blog/${featuredPost.slug}`} className="hover:text-blue-600">
                      {featuredPost.title}
                    </Link>
                  </h2>
                  <p className="text-gray-600 mb-4">{featuredPost.excerpt}</p>
                  <div className="flex items-center text-sm text-gray-500 mb-4">
                    <span>{featuredPost.author}</span>
                    <span className="mx-2">•</span>
                    <span>{featuredPost.date}</span>
                    <span className="mx-2">•</span>
                    <span>{featuredPost.readTime}</span>
                  </div>
                  <Link
                    href={`/blog/${featuredPost.slug}`}
                    className="text-blue-600 font-semibold hover:text-blue-800"
                  >
                    Read More →
                  </Link>
                </div>
              </div>
            </div>
          </div>

          {/* Categories */}
          <div className="mb-8">
            <div className="flex flex-wrap gap-2 justify-center">
              {categories.map(category => (
                <button
                  key={category}
                  className="px-4 py-2 rounded-full text-sm font-medium bg-gray-200 text-gray-700 hover:bg-blue-100 hover:text-blue-700 transition-colors"
                >
                  {category}
                </button>
              ))}
            </div>
          </div>

          {/* Blog Posts Grid */}
          <div className="grid md:grid-cols-2 lg:grid-cols-3 gap-8">
            {blogPosts.map((post, index) => (
              <article key={index} className="bg-white rounded-lg shadow-md overflow-hidden hover:shadow-lg transition-shadow">
                <Link href={`/blog/${post.slug}`}>
                  <Image
                    src={post.image}
                    alt={post.title}
                    width={400}
                    height={250}
                    className="w-full h-48 object-cover hover:scale-105 transition-transform duration-300"
                  />
                </Link>
                <div className="p-6">
                  <div className="mb-3">
                    <span className="bg-gray-100 text-gray-600 text-xs font-medium px-2 py-1 rounded">
                      {post.category}
                    </span>
                  </div>
                  <h3 className="text-lg font-bold text-gray-900 mb-3">
                    <Link href={`/blog/${post.slug}`} className="hover:text-blue-600">
                      {post.title}
                    </Link>
                  </h3>
                  <p className="text-gray-600 text-sm mb-4">{post.excerpt}</p>
                  <div className="flex items-center text-xs text-gray-500 mb-3">
                    <span>{post.author}</span>
                    <span className="mx-2">•</span>
                    <span>{post.date}</span>
                    <span className="mx-2">•</span>
                    <span>{post.readTime}</span>
                  </div>
                  <Link
                    href={`/blog/${post.slug}`}
                    className="text-blue-600 text-sm font-semibold hover:text-blue-800"
                  >
                    Read More →
                  </Link>
                </div>
              </article>
            ))}
          </div>

          {/* Newsletter Signup */}
          <div className="mt-16 bg-blue-50 rounded-lg p-8 text-center">
            <h3 className="text-2xl font-bold text-gray-900 mb-4">
              Stay Updated with Insurance Tips
            </h3>
            <p className="text-gray-600 mb-6">
              Get the latest insurance insights, money-saving tips, and expert advice delivered to your inbox.
            </p>
            <div className="max-w-md mx-auto flex gap-4">
              <input
                type="email"
                placeholder="Enter your email"
                className="flex-1 px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
              />
              <button className="bg-blue-600 text-white px-6 py-2 rounded-lg font-semibold hover:bg-blue-700 transition-colors">
                Subscribe
              </button>
            </div>
          </div>
        </div>
      </div>
    </div>
  )
}
